/**
 * Segment Detection & CTA Management
 * Detects user persona (DIY vs Agency) and shows appropriate CTAs
 */

(function() {
    'use strict';
    
    let currentSegment = 'diy'; // Default to DIY
    
    /**
     * Initialize segmentation
     */
    function initSegmentation() {
        // Try to get segment from various sources
        currentSegment = detectSegment();
        
        // Store in localStorage for persistence
        localStorage.setItem('user_segment', currentSegment);
        
        // Apply segment to page
        applySegment(currentSegment);
        
        // Track segment for analytics
        if (window.track) {
            track('segment_detected', {
                segment: currentSegment,
                source: getSegmentSource()
            });
        }
    }
    
    /**
     * Detect user segment from various sources
     */
    function detectSegment() {
        // 1. Check URL parameter ?seg=
        const urlParams = new URLSearchParams(window.location.search);
        const urlSegment = urlParams.get('seg');
        if (urlSegment) {
            if (urlSegment === 'agency' || urlSegment === 'diy') {
                return urlSegment;
            }
        }
        
        // 2. Check UTM campaign for agency indicators
        const utmCampaign = urlParams.get('utm_campaign');
        if (utmCampaign && utmCampaign.toLowerCase().includes('agency')) {
            return 'agency';
        }
        
        // 3. Check referrer for agency-related sites
        const referrer = document.referrer.toLowerCase();
        const agencyIndicators = [
            'agency',
            'marketing-agency', 
            'digital-agency',
            'ghl-agency',
            'white-label'
        ];
        
        for (const indicator of agencyIndicators) {
            if (referrer.includes(indicator)) {
                return 'agency';
            }
        }
        
        // 4. Check localStorage for previous segment
        const storedSegment = localStorage.getItem('user_segment');
        if (storedSegment && (storedSegment === 'agency' || storedSegment === 'diy')) {
            return storedSegment;
        }
        
        // 5. Default to DIY
        return 'diy';
    }
    
    /**
     * Get the source of segment detection for analytics
     */
    function getSegmentSource() {
        const urlParams = new URLSearchParams(window.location.search);
        
        if (urlParams.get('seg')) return 'url_parameter';
        if (urlParams.get('utm_campaign')?.toLowerCase().includes('agency')) return 'utm_campaign';
        if (document.referrer.toLowerCase().includes('agency')) return 'referrer';
        if (localStorage.getItem('user_segment')) return 'localStorage';
        return 'default';
    }
    
    /**
     * Apply segment to page elements
     */
    function applySegment(segment) {
        // Hide all CTA variants first
        document.querySelectorAll('[data-cta]').forEach(cta => {
            cta.style.display = 'none';
        });
        
        // Show CTAs for current segment
        document.querySelectorAll(`[data-cta="${segment}"]`).forEach(cta => {
            cta.style.display = 'block';
        });
        
        // Add segment class to body for CSS targeting
        document.body.classList.remove('segment-diy', 'segment-agency');
        document.body.classList.add(`segment-${segment}`);
        
        // Update any segment-specific content
        updateSegmentContent(segment);
    }
    
    /**
     * Update content based on segment
     */
    function updateSegmentContent(segment) {
        // Update any dynamic content based on segment
        const segmentElements = document.querySelectorAll('[data-segment-content]');
        
        segmentElements.forEach(element => {
            const content = element.dataset.segmentContent;
            try {
                const contentObj = JSON.parse(content);
                if (contentObj[segment]) {
                    element.textContent = contentObj[segment];
                }
            } catch (e) {
                // Invalid JSON, skip
            }
        });
    }
    
    /**
     * Manually set segment (for testing or user preference)
     */
    function setSegment(segment) {
        if (segment !== 'diy' && segment !== 'agency') {
            console.warn('Invalid segment:', segment);
            return;
        }
        
        currentSegment = segment;
        localStorage.setItem('user_segment', segment);
        applySegment(segment);
        
        // Track manual segment change
        if (window.track) {
            track('segment_changed', {
                segment: segment,
                source: 'manual'
            });
        }
    }
    
    /**
     * Get current segment
     */
    function getCurrentSegment() {
        return currentSegment;
    }
    
    /**
     * Track CTA interactions
     */
    function trackCTAInteraction(element) {
        const segment = getCurrentSegment();
        const ctaType = element.dataset.cta || 'unknown';
        const ctaText = element.textContent.trim();
        const ctaUrl = element.href;
        
        if (window.track) {
            track('cta_interaction', {
                segment: segment,
                cta_type: ctaType,
                cta_text: ctaText,
                cta_url: ctaUrl
            });
        }
    }
    
    /**
     * Set up event listeners for CTA tracking
     */
    function setupCTATracking() {
        document.addEventListener('click', function(e) {
            const cta = e.target.closest('[data-cta] a, [data-cta] button');
            if (cta) {
                trackCTAInteraction(cta);
            }
        });
    }
    
    /**
     * Initialize on DOM ready
     */
    function init() {
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', function() {
                initSegmentation();
                setupCTATracking();
            });
        } else {
            initSegmentation();
            setupCTATracking();
        }
    }
    
    // Expose public functions
    window.Segment = {
        setSegment: setSegment,
        getCurrentSegment: getCurrentSegment,
        init: init
    };
    
    // Auto-initialize
    init();
    
})();

// Convenience function for testing segments
function testSegment(segment) {
    if (window.Segment) {
        window.Segment.setSegment(segment);
        console.log('Segment set to:', segment);
    }
}
