/**
 * Analytics Helper for GHL Toolset
 * Clean event tracking API with consent gating
 */

(() => {
  'use strict';

  // Event queue for events triggered before consent/scripts load
  let eventQueue = [];
  let analyticsReady = false;

  /**
   * Main tracking function
   * Only sends events if analytics consent is given
   */
  window.track = function(eventName, params = {}) {
    // Check consent
    if (!window.__consent || !window.__consent.analytics) {
      console.debug('Analytics tracking blocked - no consent');
      return;
    }

    // Enhance params with common data
    const enhancedParams = {
      ...params,
      page_title: document.title,
      page_location: window.location.href,
      page_referrer: document.referrer,
      user_agent: navigator.userAgent,
      screen_resolution: `${screen.width}x${screen.height}`,
      viewport_size: `${window.innerWidth}x${window.innerHeight}`,
      timestamp: Date.now(),
      session_id: getSessionId(),
      user_id: getUserId()
    };

    // Add tool context if available
    const toolName = getCurrentTool();
    if (toolName) {
      enhancedParams.tool_name = toolName;
    }

    // Queue event if GA4 not ready yet
    if (!analyticsReady || !window.gtag) {
      eventQueue.push({ eventName, params: enhancedParams });
      console.debug(`Queued event: ${eventName}`, enhancedParams);
      return;
    }

    // Send to GA4
    try {
      window.gtag('event', eventName, enhancedParams);
      console.debug(`Tracked event: ${eventName}`, enhancedParams);
    } catch (error) {
      console.error('Failed to track event:', error);
    }
  };

  /**
   * Track page view
   */
  window.trackPageView = function(customParams = {}) {
    track('page_view', {
      ...customParams,
      engagement_time_msec: getTimeOnPage()
    });
  };

  /**
   * Track tool usage
   */
  window.trackToolUsage = function(toolName, action, params = {}) {
    track('tool_usage', {
      tool_name: toolName,
      tool_action: action,
      ...params
    });
  };

  /**
   * Track conversion events
   */
  window.trackConversion = function(type, value = 0, params = {}) {
    track('conversion', {
      conversion_type: type,
      conversion_value: value,
      currency: 'USD',
      ...params
    });
  };

  /**
   * Track user engagement
   */
  window.trackEngagement = function(type, params = {}) {
    track('engagement', {
      engagement_type: type,
      engagement_time_msec: getTimeOnPage(),
      ...params
    });
  };

  /**
   * Track errors
   */
  window.trackError = function(error, context = {}) {
    track('exception', {
      description: error.message || error,
      fatal: false,
      error_stack: error.stack || '',
      error_context: JSON.stringify(context),
      page_location: window.location.href
    });
  };

  /**
   * Get or generate session ID
   */
  function getSessionId() {
    let sessionId = sessionStorage.getItem('analytics_session_id');
    if (!sessionId) {
      sessionId = generateId();
      sessionStorage.setItem('analytics_session_id', sessionId);
    }
    return sessionId;
  }

  /**
   * Get or generate user ID
   */
  function getUserId() {
    let userId = localStorage.getItem('analytics_user_id');
    if (!userId) {
      userId = generateId();
      localStorage.setItem('analytics_user_id', userId);
    }
    return userId;
  }

  /**
   * Generate unique ID
   */
  function generateId() {
    return Date.now().toString(36) + Math.random().toString(36).substr(2);
  }

  /**
   * Get current tool name from URL
   */
  function getCurrentTool() {
    const path = window.location.pathname;
    const filename = path.split('/').pop();
    
    if (filename && filename.includes('.php')) {
      return filename.replace('.php', '');
    }
    
    return 'homepage';
  }

  /**
   * Calculate time spent on page
   */
  function getTimeOnPage() {
    if (!window.pageLoadTime) {
      window.pageLoadTime = Date.now();
    }
    return Date.now() - window.pageLoadTime;
  }

  /**
   * Process queued events when analytics becomes ready
   */
  function processEventQueue() {
    if (eventQueue.length > 0) {
      console.debug(`Processing ${eventQueue.length} queued events`);
      
      eventQueue.forEach(({ eventName, params }) => {
        try {
          window.gtag('event', eventName, params);
        } catch (error) {
          console.error('Failed to process queued event:', error);
        }
      });
      
      eventQueue = [];
    }
  }

  /**
   * Initialize analytics when consent is given
   */
  function initAnalytics() {
    // Wait for GA4 to be ready
    const checkGA4Ready = setInterval(() => {
      if (window.gtag && window.__scriptsLoaded && window.__scriptsLoaded.ga4) {
        clearInterval(checkGA4Ready);
        analyticsReady = true;
        
        // Process any queued events
        processEventQueue();
        
        // Track initial page view
        trackPageView({
          page_load_time: Date.now() - (window.pageLoadTime || Date.now())
        });
        
        console.debug('Analytics ready');
      }
    }, 100);

    // Timeout after 10 seconds
    setTimeout(() => {
      if (!analyticsReady) {
        clearInterval(checkGA4Ready);
        console.warn('Analytics initialization timeout');
      }
    }, 10000);
  }

  /**
   * Handle consent changes
   */
  function handleConsentChange(event) {
    const { current } = event.detail;
    
    if (current.analytics && !analyticsReady) {
      initAnalytics();
    } else if (!current.analytics && analyticsReady) {
      // Consent revoked
      analyticsReady = false;
      eventQueue = [];
      console.debug('Analytics disabled - consent revoked');
    }
  }

  /**
   * Track scroll depth
   */
  function trackScrollDepth() {
    let maxScroll = 0;
    const milestones = [25, 50, 75, 90, 100];
    const trackedMilestones = new Set();

    function updateScrollDepth() {
      const scrollPercent = Math.round(
        (window.scrollY / (document.body.scrollHeight - window.innerHeight)) * 100
      );
      
      if (scrollPercent > maxScroll) {
        maxScroll = scrollPercent;
        
        // Track milestone
        milestones.forEach(milestone => {
          if (scrollPercent >= milestone && !trackedMilestones.has(milestone)) {
            trackedMilestones.add(milestone);
            track('scroll_depth', {
              scroll_percent: milestone,
              max_scroll_percent: maxScroll
            });
          }
        });
      }
    }

    // Throttle scroll tracking
    let scrollTimeout;
    window.addEventListener('scroll', () => {
      if (scrollTimeout) return;
      
      scrollTimeout = setTimeout(() => {
        updateScrollDepth();
        scrollTimeout = null;
      }, 250);
    });
  }

  /**
   * Track click events on important elements
   */
  function trackClicks() {
    document.addEventListener('click', (event) => {
      const target = event.target.closest('a, button, [data-track]');
      if (!target) return;

      const trackingData = {
        element_type: target.tagName.toLowerCase(),
        element_text: target.textContent?.trim().substring(0, 100) || '',
        element_href: target.href || '',
        element_class: target.className || '',
        element_id: target.id || ''
      };

      // Track affiliate links
      if (target.classList.contains('affiliate-link')) {
        track('click_affiliate', {
          ...trackingData,
          partner_id: target.dataset.partner || 'unknown'
        });
      }
      // Track CTA buttons
      else if (target.classList.contains('cta-link') || target.dataset.track === 'cta') {
        track('click_cta', {
          ...trackingData,
          cta_type: target.dataset.ctaType || 'generic'
        });
      }
      // Track navigation links
      else if (target.classList.contains('nav-link')) {
        track('click_navigation', trackingData);
      }
      // Track generic buttons
      else if (target.tagName === 'BUTTON') {
        track('click_button', trackingData);
      }
      // Track external links
      else if (target.href && !target.href.includes(window.location.hostname)) {
        track('click_external_link', trackingData);
      }
    });
  }

  /**
   * Track form submissions
   */
  function trackForms() {
    document.addEventListener('submit', (event) => {
      const form = event.target;
      if (!form.tagName || form.tagName !== 'FORM') return;

      track('form_submit', {
        form_id: form.id || '',
        form_class: form.className || '',
        form_action: form.action || '',
        form_method: form.method || 'get',
        field_count: form.elements.length
      });
    });
  }

  /**
   * Track copy to clipboard events
   */
  function trackCopyEvents() {
    // Listen for custom copy events
    document.addEventListener('copyToClipboard', (event) => {
      track('copy_to_clipboard', {
        content_type: event.detail?.type || 'text',
        content_length: event.detail?.content?.length || 0,
        tool_name: getCurrentTool()
      });
    });
  }

  /**
   * Track Core Web Vitals
   */
  function trackWebVitals() {
    // Only load web-vitals if analytics consent is given
    if (!window.__consent?.analytics) return;

    // Dynamically import web-vitals
    import('https://unpkg.com/web-vitals@3/dist/web-vitals.js').then(({ onCLS, onFID, onFCP, onLCP, onTTFB, onINP }) => {
      onCLS((metric) => track('web_vital', { metric_name: 'CLS', metric_value: metric.value }));
      onFID((metric) => track('web_vital', { metric_name: 'FID', metric_value: metric.value }));
      onFCP((metric) => track('web_vital', { metric_name: 'FCP', metric_value: metric.value }));
      onLCP((metric) => track('web_vital', { metric_name: 'LCP', metric_value: metric.value }));
      onTTFB((metric) => track('web_vital', { metric_name: 'TTFB', metric_value: metric.value }));
      onINP((metric) => track('web_vital', { metric_name: 'INP', metric_value: metric.value }));
    }).catch(console.error);
  }

  /**
   * Initialize tracking
   */
  function init() {
    // Set page load time
    window.pageLoadTime = Date.now();

    // Track unhandled errors
    window.addEventListener('error', (event) => {
      trackError(event.error || new Error(event.message), {
        filename: event.filename,
        lineno: event.lineno,
        colno: event.colno
      });
    });

    // Track unhandled promise rejections
    window.addEventListener('unhandledrejection', (event) => {
      trackError(event.reason, { type: 'unhandled_promise_rejection' });
    });

    // Set up event listeners
    trackScrollDepth();
    trackClicks();
    trackForms();
    trackCopyEvents();

    // Listen for consent changes
    window.addEventListener('consentChanged', handleConsentChange);
    window.addEventListener('consentReady', (event) => {
      if (event.detail.analytics) {
        initAnalytics();
        trackWebVitals();
      }
    });

    // Track page visibility changes
    document.addEventListener('visibilitychange', () => {
      track('page_visibility', {
        visibility_state: document.visibilityState,
        time_on_page: getTimeOnPage()
      });
    });

    // Track page unload
    window.addEventListener('beforeunload', () => {
      track('page_unload', {
        time_on_page: getTimeOnPage(),
        max_scroll_percent: Math.round(
          (window.scrollY / (document.body.scrollHeight - window.innerHeight)) * 100
        )
      });
    });
  }

  // Initialize when DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }

})();