<?php
/**
 * CTA Component for GHL Toolset
 * Persona-based CTAs with conversion tracking
 */

// CTA configurations for different personas
$CTAS = [
    'diy' => [
        'headline' => 'Make GHL actually work for you',
        'subheadline' => 'Stop paying for features you never deploy. Get a 1:1 audit with a GHL expert.',
        'bullets' => [
            'Quick wins you can implement in 7 days',
            'Fix misconfigurations costing you leads',
            'Custom roadmap you can actually execute'
        ],
        'button_text' => 'Book Free Audit',
        'urgency' => 'Only 3 spots available this week',
        'social_proof' => '127 GHL users have already improved their setup',
        'value_prop' => '$2,000+ value audit — completely free'
    ],
    'agency' => [
        'headline' => 'Your agency should do better',
        'subheadline' => 'We\'ll audit your GHL account and show the gaps your agency isn\'t closing.',
        'bullets' => [
            'Pipeline cleanup that converts 40% more leads',
            'Automations that actually work reliably',
            'Reporting that shows real business impact'
        ],
        'button_text' => 'See What You\'re Missing',
        'urgency' => 'Most agencies miss 60% of GHL\'s potential',
        'social_proof' => '89% of our audits find critical gaps',
        'value_prop' => 'Find out what you\'re not getting'
    ],
    'frustrated' => [
        'headline' => 'GHL doesn\'t have to be this hard',
        'subheadline' => 'If you\'re struggling with setup, automation, or getting results — we can help.',
        'bullets' => [
            'End the constant troubleshooting',
            'Get systems that work the first time',
            'Stop leaving money on the table'
        ],
        'button_text' => 'Get Help Now',
        'urgency' => 'Every day you wait is revenue lost',
        'social_proof' => 'Helped 200+ businesses fix their GHL',
        'value_prop' => 'Turn frustration into results'
    ]
];

/**
 * Determine user persona based on context
 */
function detect_persona() {
    // Check URL parameters first
    if (isset($_GET['seg'])) {
        $segment = $_GET['seg'];
        if (array_key_exists($segment, $GLOBALS['CTAS'])) {
            return $segment;
        }
    }
    
    // Check referrer patterns
    $referrer = $_SERVER['HTTP_REFERER'] ?? '';
    if (strpos($referrer, 'agency') !== false || strpos($referrer, 'service') !== false) {
        return 'agency';
    }
    
    // Check user agent for mobile (DIY users often mobile)
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    if (preg_match('/Mobile|Android|iPhone/', $userAgent)) {
        return 'diy';
    }
    
    // Check visited tools (will be implemented via JavaScript)
    // For now, default to DIY
    return 'diy';
}

/**
 * Get CTA configuration for persona
 */
function get_cta_config($persona = null) {
    global $CTAS;
    
    if (!$persona) {
        $persona = detect_persona();
    }
    
    return $CTAS[$persona] ?? $CTAS['diy'];
}

/**
 * Render persona-based CTA section
 */
function render_cta($persona = null, $options = []) {
    // CTA styling classes
    $ctaClass = 'cta-section';
    if (isset($options['variant'])) {
        $ctaClass .= ' cta-' . $options['variant'];
    }
    if (isset($options['placement'])) {
        $ctaClass .= ' cta-placement-' . $options['placement'];
    }
    
    ?>
    <section class="<?php echo htmlspecialchars($ctaClass); ?>">
        <!-- DIY Owner CTA -->
        <div data-cta="diy" class="cta-variant">
            <?php render_persona_cta('diy', $options); ?>
        </div>
        
        <!-- Bad Agency Client CTA -->
        <div data-cta="agency" class="cta-variant">
            <?php render_persona_cta('agency', $options); ?>
        </div>
    </section>
    
    <!-- Load segment.js for persona detection -->
    <script src="<?php echo url('lib/js/segment.js'); ?>" defer></script>
    <?php
}

/**
 * Render CTA for specific persona
 */
function render_persona_cta($persona, $options = []) {
    $config = get_cta_config($persona);
    
    // Build booking URL with UTM parameters
    $utm_params = [
        'utm_source' => DEFAULT_UTM_SOURCE,
        'utm_medium' => DEFAULT_UTM_MEDIUM,
        'utm_campaign' => 'audit',
        'utm_content' => $persona,
        'seg' => $persona
    ];
    
    // Add tool context if available
    if (function_exists('get_current_tool')) {
        $currentTool = get_current_tool();
        if ($currentTool && $currentTool !== 'homepage') {
            $utm_params['utm_term'] = $currentTool;
        }
    }
    
    // Add first-touch attribution
    if (function_exists('get_first_touch_utm')) {
        $firstTouch = get_first_touch_utm();
        if ($firstTouch) {
            $utm_params['utm_source_original'] = $firstTouch['source'];
            $utm_params['utm_campaign_original'] = $firstTouch['campaign'];
        }
    }
    
    $bookingUrl = BOOKING_URL . '?' . http_build_query($utm_params);
    
    ?>
    <div class="cta-container" data-persona="<?php echo htmlspecialchars($persona); ?>">
        <div class="cta-content">
            <div class="cta-header">
                <h2 class="cta-headline"><?php echo htmlspecialchars($config['headline']); ?></h2>
                <p class="cta-subheadline"><?php echo htmlspecialchars($config['subheadline']); ?></p>
            </div>
            
            <div class="cta-benefits">
                <ul class="cta-bullets">
                    <?php foreach ($config['bullets'] as $bullet): ?>
                        <li class="cta-bullet">
                            <span class="cta-bullet-icon">✓</span>
                            <span class="cta-bullet-text"><?php echo htmlspecialchars($bullet); ?></span>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>
            
            <div class="cta-action">
                <a href="<?php echo htmlspecialchars($bookingUrl); ?>" 
                   class="btn btn-primary btn-lg cta-button"
                   target="_blank"
                   rel="nofollow"
                   data-persona="<?php echo htmlspecialchars($persona); ?>"
                   data-placement="<?php echo htmlspecialchars($options['placement'] ?? 'content'); ?>"
                   onclick="track('click_cta', {
                       persona: '<?php echo $persona; ?>',
                       placement: '<?php echo $options['placement'] ?? 'content'; ?>',
                       tool: '<?php echo function_exists('get_current_tool') ? get_current_tool() : ''; ?>'
                   })">
                    <?php echo htmlspecialchars($config['button_text']); ?>
                </a>
                
                <div class="cta-value-prop">
                    <span class="cta-value-text"><?php echo htmlspecialchars($config['value_prop']); ?></span>
                </div>
            </div>
            
            <div class="cta-footer">
                <div class="cta-social-proof">
                    <span class="cta-social-proof-icon">👥</span>
                    <span class="cta-social-proof-text"><?php echo htmlspecialchars($config['social_proof']); ?></span>
                </div>
                
                <?php if (isset($config['urgency'])): ?>
                    <div class="cta-urgency">
                        <span class="cta-urgency-icon">⏰</span>
                        <span class="cta-urgency-text"><?php echo htmlspecialchars($config['urgency']); ?></span>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <?php if (isset($options['show_image']) && $options['show_image']): ?>
            <div class="cta-visual">
                <div class="cta-image">
                    <div class="cta-testimonial">
                        <blockquote>
                            "Paddle CRM found 12 issues in my GHL setup that my previous agency completely missed."
                        </blockquote>
                        <cite>— Sarah K., Real Estate Agent</cite>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
    
    <style>
        .cta-section {
            background: linear-gradient(135deg, var(--color-primary-50) 0%, var(--color-primary-100) 100%);
            padding: var(--space-16) 0;
            margin: var(--space-20) 0;
            border-radius: var(--radius-2xl);
            position: relative;
            overflow: hidden;
        }
        
        .cta-section::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: radial-gradient(circle at 30% 20%, rgba(99, 102, 241, 0.1) 0%, transparent 50%);
            pointer-events: none;
        }
        
        .cta-container {
            max-width: var(--container-lg);
            margin: 0 auto;
            padding: 0 var(--space-6);
            display: grid;
            grid-template-columns: 1fr;
            gap: var(--space-12);
            align-items: center;
            position: relative;
            z-index: 1;
        }
        
        .cta-content {
            text-align: center;
        }
        
        .cta-header {
            margin-bottom: var(--space-8);
        }
        
        .cta-headline {
            font-size: var(--font-size-4xl);
            font-weight: var(--font-weight-bold);
            color: var(--color-neutral-900);
            margin-bottom: var(--space-4);
            line-height: var(--line-height-tight);
        }
        
        .cta-subheadline {
            font-size: var(--font-size-xl);
            color: var(--color-neutral-700);
            line-height: var(--line-height-relaxed);
            max-width: 600px;
            margin: 0 auto;
        }
        
        .cta-benefits {
            margin-bottom: var(--space-10);
        }
        
        .cta-bullets {
            list-style: none;
            padding: 0;
            margin: 0;
            display: inline-block;
            text-align: left;
        }
        
        .cta-bullet {
            display: flex;
            align-items: center;
            margin-bottom: var(--space-3);
            font-size: var(--font-size-lg);
        }
        
        .cta-bullet-icon {
            color: var(--color-success);
            font-weight: var(--font-weight-bold);
            margin-right: var(--space-3);
            font-size: var(--font-size-xl);
        }
        
        .cta-action {
            margin-bottom: var(--space-8);
        }
        
        .cta-button {
            font-size: var(--font-size-xl);
            padding: var(--space-5) var(--space-10);
            box-shadow: var(--shadow-lg);
            margin-bottom: var(--space-4);
            display: inline-block;
        }
        
        .cta-value-prop {
            font-size: var(--font-size-lg);
            color: var(--color-primary-700);
            font-weight: var(--font-weight-semibold);
        }
        
        .cta-footer {
            display: flex;
            flex-direction: column;
            gap: var(--space-3);
            align-items: center;
        }
        
        .cta-social-proof,
        .cta-urgency {
            display: flex;
            align-items: center;
            gap: var(--space-2);
            font-size: var(--font-size-sm);
            color: var(--color-neutral-600);
        }
        
        .cta-urgency {
            color: var(--color-warning);
            font-weight: var(--font-weight-medium);
        }
        
        .cta-testimonial {
            background: white;
            padding: var(--space-6);
            border-radius: var(--radius-xl);
            box-shadow: var(--shadow-md);
            text-align: left;
            max-width: 400px;
        }
        
        .cta-testimonial blockquote {
            font-style: italic;
            margin-bottom: var(--space-4);
            font-size: var(--font-size-lg);
            color: var(--color-neutral-700);
        }
        
        .cta-testimonial cite {
            font-style: normal;
            font-weight: var(--font-weight-semibold);
            color: var(--color-neutral-900);
        }
        
        /* Responsive adjustments */
        @media (min-width: 768px) {
            .cta-container {
                grid-template-columns: 2fr 1fr;
                text-align: left;
            }
            
            .cta-content {
                text-align: left;
            }
            
            .cta-headline {
                font-size: var(--font-size-5xl);
            }
            
            .cta-footer {
                align-items: flex-start;
            }
        }
        
        /* Variant styles */
        .cta-compact {
            padding: var(--space-12) 0;
            margin: var(--space-12) 0;
        }
        
        .cta-compact .cta-headline {
            font-size: var(--font-size-3xl);
        }
        
        .cta-sidebar {
            background: var(--color-neutral-50);
            padding: var(--space-8);
            border-radius: var(--radius-xl);
            border: 1px solid var(--color-neutral-200);
            margin: var(--space-8) 0;
        }
        
        .cta-sidebar .cta-headline {
            font-size: var(--font-size-2xl);
        }
        
        /* Placement-specific styles */
        .cta-placement-sidebar {
            max-width: 400px;
        }
        
        .cta-placement-footer {
            margin-top: var(--space-24);
        }
    </style>
    <?php
}

/**
 * Render inline CTA (smaller version)
 */
function render_inline_cta($persona = null, $text = null) {
    $config = get_cta_config($persona);
    $actualPersona = $persona ?: detect_persona();
    $buttonText = $text ?: $config['button_text'];
    
    $utm_params = [
        'utm_source' => DEFAULT_UTM_SOURCE,
        'utm_medium' => DEFAULT_UTM_MEDIUM,
        'utm_campaign' => 'audit-inline',
        'utm_content' => $actualPersona,
        'seg' => $actualPersona
    ];
    
    $bookingUrl = BOOKING_URL . '?' . http_build_query($utm_params);
    
    echo '<a href="' . htmlspecialchars($bookingUrl) . '" ';
    echo 'class="cta-inline btn btn-primary" ';
    echo 'target="_blank" rel="nofollow" ';
    echo 'data-persona="' . htmlspecialchars($actualPersona) . '" ';
    echo 'onclick="track(\'click_cta\', {persona: \'' . $actualPersona . '\', placement: \'inline\'})">';
    echo htmlspecialchars($buttonText);
    echo '</a>';
}

/**
 * Render exit-intent modal CTA
 */
function render_exit_intent_cta($persona = null) {
    // Only show on audit page
    if (function_exists('get_current_tool') && get_current_tool() !== 'audit') {
        return;
    }
    
    $config = get_cta_config($persona);
    $actualPersona = $persona ?: detect_persona();
    
    ?>
    <div id="exit-intent-modal" class="exit-intent-modal" style="display: none;">
        <div class="exit-intent-backdrop"></div>
        <div class="exit-intent-content">
            <button class="exit-intent-close" onclick="closeExitIntent()">&times;</button>
            <h3>Wait! Don't leave without your free audit</h3>
            <p><?php echo htmlspecialchars($config['subheadline']); ?></p>
            <a href="<?php echo htmlspecialchars(BOOKING_URL . '?utm_source=exit-intent&seg=' . $actualPersona); ?>" 
               class="btn btn-primary btn-lg"
               target="_blank"
               rel="nofollow"
               onclick="track('click_cta', {persona: '<?php echo $actualPersona; ?>', placement: 'exit-intent'})">
                <?php echo htmlspecialchars($config['button_text']); ?>
            </a>
        </div>
    </div>
    
    <script>
        let exitIntentShown = false;
        
        function showExitIntent() {
            if (exitIntentShown) return;
            
            const modal = document.getElementById('exit-intent-modal');
            if (modal) {
                modal.style.display = 'block';
                exitIntentShown = true;
                
                // Set cookie to suppress for 14 days
                document.cookie = 'exit_intent_shown=1; max-age=' + (14 * 24 * 60 * 60) + '; path=/';
                
                track('view_exit_intent', {persona: '<?php echo $actualPersona; ?>'});
            }
        }
        
        function closeExitIntent() {
            const modal = document.getElementById('exit-intent-modal');
            if (modal) {
                modal.style.display = 'none';
            }
        }
        
        // Only enable exit intent if not shown recently
        if (!document.cookie.includes('exit_intent_shown=1')) {
            document.addEventListener('mouseleave', function(e) {
                if (e.clientY < 0) {
                    showExitIntent();
                }
            });
        }
    </script>
    
    <style>
        .exit-intent-modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            z-index: var(--z-modal);
        }
        
        .exit-intent-backdrop {
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
        }
        
        .exit-intent-content {
            position: relative;
            max-width: 500px;
            margin: 15vh auto;
            background: white;
            padding: var(--space-8);
            border-radius: var(--radius-xl);
            text-align: center;
            box-shadow: var(--shadow-2xl);
        }
        
        .exit-intent-close {
            position: absolute;
            top: var(--space-4);
            right: var(--space-4);
            background: none;
            border: none;
            font-size: var(--font-size-2xl);
            cursor: pointer;
            line-height: 1;
            color: var(--color-neutral-400);
        }
        
        .exit-intent-content h3 {
            margin-bottom: var(--space-4);
            color: var(--color-neutral-900);
        }
        
        .exit-intent-content p {
            margin-bottom: var(--space-6);
            color: var(--color-neutral-600);
        }
    </style>
    <?php
}
?>