<?php
/**
 * Ad Slot Component for GHL Toolset
 * CLS-safe ad rendering with consent gating
 */

/**
 * Render an ad slot with reserved space
 * 
 * @param string $slot Slot type: 'top', 'in-content', 'bottom', 'sidebar'
 * @param string $adSlotId AdSense ad slot ID
 * @param array $options Additional options for customization
 */
function render_ad_slot($slot = 'top', $adSlotId = '', $options = []) {
    if (!ENABLE_ADS) {
        return;
    }
    
    // Check if top ads are disabled
    if ($slot === 'top' && !SHOW_TOP_ADS) {
        return;
    }
    
    // Check frequency cap for returning users
    if (should_suppress_ads($slot)) {
        return;
    }
    
    $classes = ['ad-slot', $slot];
    if (isset($options['class'])) {
        $classes[] = $options['class'];
    }
    
    $styles = [];
    
    // Set minimum height based on slot type to prevent CLS
    switch ($slot) {
        case 'top':
            $styles[] = 'min-height: var(--ad-height-top)';
            break;
        case 'in-content':
            $styles[] = 'min-height: var(--ad-height-in-content)';
            break;
        case 'bottom':
            $styles[] = 'min-height: var(--ad-height-bottom)';
            break;
        case 'sidebar':
            $styles[] = 'min-height: var(--ad-height-in-content)';
            $styles[] = 'width: var(--ad-width-sidebar)';
            break;
    }
    
    // Add custom styles
    if (isset($options['style'])) {
        $styles[] = $options['style'];
    }
    
    $classString = implode(' ', $classes);
    $styleString = implode('; ', $styles);
    
    // Unique ID for this ad slot
    $uniqueId = 'ad-' . $slot . '-' . uniqid();
    
    echo '<div class="' . htmlspecialchars($classString) . '" 
              style="' . htmlspecialchars($styleString) . '" 
              id="' . htmlspecialchars($uniqueId) . '"
              data-slot-type="' . htmlspecialchars($slot) . '"
              data-ad-slot="' . htmlspecialchars($adSlotId) . '">';
    
    // Placeholder content before ads load
    echo '<div class="ad-placeholder">';
    echo '<div class="ad-placeholder-content">';
    echo '<span class="ad-placeholder-icon">📢</span>';
    echo '<span class="ad-placeholder-text">Advertisement</span>';
    echo '</div>';
    echo '</div>';
    
    // AdSense ad unit (will be populated by JavaScript after consent)
    echo '<ins class="adsbygoogle" 
              style="display:block; width:100%; height:100%;"
              data-ad-client="' . htmlspecialchars(ADSENSE_PUB) . '"
              data-ad-slot="' . htmlspecialchars($adSlotId) . '"
              data-ad-format="' . get_ad_format($slot) . '"
              data-full-width-responsive="true"
              data-adtest="' . (defined('ADSENSE_TEST_MODE') && ADSENSE_TEST_MODE ? 'on' : 'off') . '"></ins>';
    
    echo '</div>';
    
    // Add analytics tracking for ad impression
    echo '<script>';
    echo 'document.addEventListener("DOMContentLoaded", function() {';
    echo '  if (window.__consent && window.__consent.ads) {';
    echo '    track("ad_impression", {';
    echo '      slot_type: "' . $slot . '",';
    echo '      slot_id: "' . $adSlotId . '",';
    echo '      ad_unit_id: "' . $uniqueId . '"';
    echo '    });';
    echo '  }';
    echo '});';
    echo '</script>';
}

/**
 * Get appropriate ad format for slot type
 */
function get_ad_format($slot) {
    switch ($slot) {
        case 'top':
        case 'bottom':
            return 'horizontal';
        case 'sidebar':
            return 'rectangle';
        case 'in-content':
            return 'fluid';
        default:
            return 'auto';
    }
}

/**
 * Check if ads should be suppressed based on frequency cap
 */
function should_suppress_ads($slot) {
    if ($slot !== 'in-content') {
        return false; // Only apply frequency cap to in-content ads
    }
    
    $lastAdView = $_COOKIE['last_ad_view'] ?? 0;
    $hoursSinceLastView = (time() - $lastAdView) / 3600;
    
    return $hoursSinceLastView < AD_FREQUENCY_CAP_HOURS;
}

/**
 * Set ad view timestamp for frequency capping
 */
function set_ad_view_timestamp() {
    setcookie('last_ad_view', time(), time() + (AD_FREQUENCY_CAP_HOURS * 3600), '/');
}

/**
 * Render ad blocker detection and fallback
 */
function render_ad_fallback($slot, $options = []) {
    echo '<div class="ad-fallback" style="display: none;" data-slot="' . htmlspecialchars($slot) . '">';
    
    if (isset($options['fallback_content'])) {
        echo $options['fallback_content'];
    } else {
        // Default fallback content
        echo '<div class="ad-fallback-content">';
        echo '<h4>Support Our Free Tools</h4>';
        echo '<p>We rely on ads to keep these tools free. Please consider:</p>';
        echo '<ul>';
        echo '<li>Whitelisting our site in your ad blocker</li>';
        echo '<li><a href="' . BOOKING_URL . '?utm_source=ad-fallback&utm_medium=site" target="_blank" rel="nofollow">Booking a consultation</a> with our team</li>';
        echo '<li>Sharing our tools with colleagues</li>';
        echo '</ul>';
        echo '</div>';
    }
    
    echo '</div>';
}

/**
 * Render multiple ad slots for a page layout
 */
function render_page_ads($layout = 'standard', $adSlots = []) {
    switch ($layout) {
        case 'tool-page':
            if (isset($adSlots['top'])) {
                render_ad_slot('top', $adSlots['top']);
            }
            if (isset($adSlots['in-content'])) {
                render_ad_slot('in-content', $adSlots['in-content']);
            }
            if (isset($adSlots['bottom'])) {
                render_ad_slot('bottom', $adSlots['bottom']);
            }
            break;
            
        case 'homepage':
            if (isset($adSlots['top'])) {
                render_ad_slot('top', $adSlots['top']);
            }
            if (isset($adSlots['bottom'])) {
                render_ad_slot('bottom', $adSlots['bottom']);
            }
            break;
            
        case 'with-sidebar':
            if (isset($adSlots['top'])) {
                render_ad_slot('top', $adSlots['top']);
            }
            if (isset($adSlots['sidebar'])) {
                render_ad_slot('sidebar', $adSlots['sidebar']);
            }
            if (isset($adSlots['bottom'])) {
                render_ad_slot('bottom', $adSlots['bottom']);
            }
            break;
    }
}

/**
 * Get ad slot configuration for a specific tool
 */
function get_tool_ad_slots($toolName) {
    // This could be moved to a configuration file or database
    $toolAdSlots = [
        'sms-counter' => [
            'top' => '1234567890',
            'in-content' => '1234567891',
            'bottom' => '1234567892'
        ],
        'roi-calculator' => [
            'top' => '1234567893',
            'in-content' => '1234567894',
            'bottom' => '1234567895'
        ],
        'utm-builder' => [
            'top' => '1234567896',
            'in-content' => '1234567897',
            'bottom' => '1234567898'
        ],
        // Add more tools as needed
    ];
    
    return $toolAdSlots[$toolName] ?? [
        'top' => '0000000000', // Default/test slot IDs
        'in-content' => '0000000001',
        'bottom' => '0000000002'
    ];
}

/**
 * Initialize ad view tracking
 */
function init_ad_tracking() {
    if (ENABLE_ADS && isset($_COOKIE['consent'])) {
        $consent = json_decode($_COOKIE['consent'], true);
        if ($consent && $consent['ads']) {
            set_ad_view_timestamp();
        }
    }
}

// CSS for ad placeholders and fallbacks
function render_ad_styles() {
    echo '<style>';
    echo '.ad-placeholder {';
    echo '  display: flex;';
    echo '  align-items: center;';
    echo '  justify-content: center;';
    echo '  background: var(--color-neutral-50);';
    echo '  border: 1px dashed var(--color-neutral-200);';
    echo '  border-radius: var(--radius-base);';
    echo '  height: 100%;';
    echo '  min-height: inherit;';
    echo '  color: var(--color-neutral-400);';
    echo '  font-size: var(--font-size-sm);';
    echo '}';
    echo '.ad-placeholder-content {';
    echo '  text-align: center;';
    echo '}';
    echo '.ad-placeholder-icon {';
    echo '  display: block;';
    echo '  font-size: var(--font-size-xl);';
    echo '  margin-bottom: var(--space-2);';
    echo '}';
    echo '.ad-fallback {';
    echo '  padding: var(--space-6);';
    echo '  background: var(--color-neutral-50);';
    echo '  border: 1px solid var(--color-neutral-200);';
    echo '  border-radius: var(--radius-lg);';
    echo '  text-align: center;';
    echo '}';
    echo '.ad-fallback h4 {';
    echo '  color: var(--color-neutral-700);';
    echo '  margin-bottom: var(--space-4);';
    echo '}';
    echo '.ad-fallback ul {';
    echo '  text-align: left;';
    echo '  max-width: 300px;';
    echo '  margin: 0 auto;';
    echo '}';
    echo '</style>';
}

// Auto-initialize tracking if called
if (basename($_SERVER['SCRIPT_NAME']) !== 'ad-slot.php') {
    init_ad_tracking();
}
?>