﻿<?php
require_once __DIR__ . '/config.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Color Palette Generator for GoHighLevel | Free Color Scheme Tool</title>
    <meta name="description" content="Free color palette generator for GoHighLevel users. Create professional color schemes for your GHL funnels, websites, and marketing materials. Perfect for agencies and designers.">
    <meta name="keywords" content="color palette generator, color scheme generator, color palette tool, GoHighLevel design, funnel colors, website colors, brand colors, color combinations, design tools">
    <meta name="author" content="Paddle CRM">
    <meta name="robots" content="index, follow">
    
    <!-- Open Graph / Facebook -->
    <meta property="og:type" content="website">
    <meta property="og:title" content="Color Palette Generator for GoHighLevel">
    <meta property="og:description" content="Free color palette generator for GoHighLevel users. Create professional color schemes for your funnels and websites.">
    <meta property="og:url" content="https://yourdomain.com/color-palette-generator.php">
    <meta property="og:image" content="https://yourdomain.com/images/color-palette-generator-og.jpg">
    
    <!-- Twitter -->
    <meta property="twitter:card" content="summary_large_image">
    <meta property="twitter:title" content="Color Palette Generator for GoHighLevel">
    <meta property="twitter:description" content="Free color palette generator for GoHighLevel users. Create professional color schemes for your funnels and websites.">
    <meta property="twitter:image" content="https://yourdomain.com/images/color-palette-generator-og.jpg">
    
    <!-- Schema.org structured data -->
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "WebApplication",
        "name": "Color Palette Generator for GoHighLevel",
        "description": "Free tool to generate professional color palettes for websites, funnels, and marketing materials",
        "url": "https://yourdomain.com/color-palette-generator.php",
        "applicationCategory": "DesignApplication",
        "operatingSystem": "Web Browser",
        "offers": {
            "@type": "Offer",
            "price": "0",
            "priceCurrency": "USD"
        },
        "provider": {
            "@type": "Organization",
            "name": "Paddle CRM",
            "description": "WhiteLabel GoHighLevel Agency"
        }
    }
    </script>
    
    <link rel="stylesheet" href="styles.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <?php inc('includes/header.php'); ?>

    <!-- Hero Section -->
    <section class="hero tool-hero">
        <div class="container">
            <nav class="breadcrumb">
                <a href="index.php">Home</a> > <span>Color Palette Generator</span>
            </nav>
            <div class="hero-content">
                <div class="hero-text">
                    <h1 class="hero-title">Color Palette Generator for GoHighLevel</h1>
                    <p class="hero-subtitle">Create professional color schemes for your GHL funnels, websites, and marketing materials</p>
                    <div class="hero-stats">
                        <div class="stat-item">
                            <span class="stat-number">Professional</span>
                            <span class="stat-label">Quality</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-number">GoHighLevel</span>
                            <span class="stat-label">Optimized</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-number">100%</span>
                            <span class="stat-label">Free Tool</span>
                        </div>
                    </div>
                </div>
                <div class="hero-visual">
                    <div class="hero-image">
                        <div class="floating-card card-1">
                            <span class="card-icon">🎨</span>
                            <span class="card-text">Colors</span>
                        </div>
                        <div class="floating-card card-2">
                            <span class="card-icon">⚡</span>
                            <span class="card-text">Fast</span>
                        </div>
                        <div class="floating-card card-3">
                            <span class="card-icon">🎯</span>
                            <span class="card-text">Harmonious</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Tool Section -->
    <section class="tool-section">
        <div class="container">
            <div class="tool-card featured">
                <div class="tool-header">
                    <div class="tool-icon">🎨</div>
                    <h2>Color Palette Generator</h2>
                    <p>Professional color scheme tool designed specifically for GoHighLevel users and agencies</p>
                </div>
                
                <div class="tool-interface">
                    <div class="input-group">
                        <label for="primaryColor">Primary Color (Hex Code)</label>
                        <input type="color" id="primaryColor" value="#0a2038">
                        <input type="text" id="primaryColorHex" placeholder="#0a2038" pattern="^#[0-9A-Fa-f]{6}$">
                    </div>
                    
                    <div class="button-group">
                        <button class="btn btn-primary" onclick="generatePalette()">Generate Color Palette</button>
                        <button class="btn btn-secondary" onclick="generateRandomPalette()">Random Palette</button>
                        <button class="btn btn-secondary" onclick="generateAccessiblePalette()">Accessible Palette</button>
                    </div>
                    
                    <div class="palette-result" id="paletteResult">
                        <h3>Your Color Palette</h3>
                        <div class="color-swatches" id="colorSwatches"></div>
                        <div class="palette-info" id="paletteInfo"></div>
                        <button class="btn btn-primary" onclick="exportPalette()">Export Palette</button>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Content Section -->
    <section class="content-section">
        <div class="container">
            <div class="content-grid">
                <article class="content-card">
                    <h2>What is a Color Palette Generator?</h2>
                    <p>A color palette generator is a tool that creates harmonious color combinations based on color theory principles. It helps designers and marketers create visually appealing and professional color schemes for websites, funnels, and marketing materials.</p>
                    
                    <p>For GoHighLevel users, proper color selection is crucial for:</p>
                    <ul>
                        <li><strong>Brand Consistency:</strong> Maintaining uniform colors across all client materials</li>
                        <li><strong>User Experience:</strong> Creating visually appealing and engaging funnels</li>
                        <li><strong>Conversion Optimization:</strong> Using colors that drive action and engagement</li>
                        <li><strong>Professional Appearance:</strong> Ensuring all client work looks polished and credible</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Why GoHighLevel Users Need This Tool</h2>
                    <p>As a GoHighLevel agency, you're constantly creating funnels, landing pages, and marketing materials for multiple clients. Each project requires a unique and professional color scheme that aligns with the client's brand and goals.</p>
                    
                    <ul>
                        <li><strong>Time Efficiency:</strong> Generate professional color schemes instantly instead of manual selection</li>
                        <li><strong>Color Theory Expertise:</strong> Access to scientifically proven color combinations</li>
                        <li><strong>Brand Consistency:</strong> Create cohesive color schemes across all client materials</li>
                        <li><strong>Professional Quality:</strong> Ensure all designs meet industry standards</li>
                        <li><strong>Client Satisfaction:</strong> Deliver visually stunning and effective designs</li>
                        <li><strong>Competitive Advantage:</strong> Stand out with superior visual design</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>How to Use the Color Palette Generator</h2>
                    <p>Our color palette generator is designed to be intuitive and powerful:</p>
                    
                    <ol>
                        <li><strong>Choose Your Primary Color:</strong> Select a base color using the color picker or enter a hex code</li>
                        <li><strong>Generate Options:</strong> Click "Generate Color Palette" to create a harmonious scheme</li>
                        <li><strong>Explore Variations:</strong> Try different generation methods:
                            <ul>
                                <li><strong>Standard Palette:</strong> Creates complementary and analogous colors</li>
                                <li><strong>Random Palette:</strong> Generates unexpected but harmonious combinations</li>
                                <li><strong>Accessible Palette:</strong> Ensures colors meet accessibility standards</li>
                            </ul>
                        </li>
                        <li><strong>Review Results:</strong> Examine the generated colors and their hex codes</li>
                        <li><strong>Export & Apply:</strong> Copy the colors to use in your GoHighLevel projects</li>
                    </ol>
                </article>

                <article class="content-card">
                    <h2>Use Cases for GoHighLevel Agencies</h2>
                    <p>This tool serves multiple purposes in your daily GoHighLevel operations:</p>
                    
                    <ul>
                        <li><strong>Funnel Design:</strong> Create cohesive color schemes for entire sales funnels</li>
                        <li><strong>Landing Page Creation:</strong> Design visually appealing landing pages with proper color psychology</li>
                        <li><strong>Email Campaigns:</strong> Develop consistent color themes for email marketing</li>
                        <li><strong>Social Media Graphics:</strong> Create branded visual content with harmonious colors</li>
                        <li><strong>Client Branding:</strong> Develop unique color identities for client businesses</li>
                        <li><strong>Website Design:</strong> Create professional website color schemes</li>
                        <li><strong>Marketing Materials:</strong> Design brochures, flyers, and other collateral</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Color Psychology & Marketing</h2>
                    <p>Understanding color psychology helps you create more effective marketing materials:</p>
                    
                    <ul>
                        <li><strong>Blue:</strong> Trust, professionalism, stability - perfect for financial and business services</li>
                        <li><strong>Green:</strong> Growth, health, money - ideal for health, finance, and environmental businesses</li>
                        <li><strong>Red:</strong> Energy, urgency, passion - great for sales, food, and entertainment</li>
                        <li><strong>Yellow:</strong> Optimism, creativity, warmth - perfect for creative and educational services</li>
                        <li><strong>Purple:</strong> Luxury, creativity, wisdom - ideal for premium and creative services</li>
                        <li><strong>Orange:</strong> Enthusiasm, adventure, confidence - great for fitness and adventure businesses</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Integration with GoHighLevel</h2>
                    <p>This tool integrates seamlessly with your GoHighLevel workflow:</p>
                    
                    <ul>
                        <li><strong>Funnel Builder:</strong> Apply generated colors to your GHL funnel designs</li>
                        <li><strong>Template Creation:</strong> Create branded templates with consistent color schemes</li>
                        <li><strong>Client Projects:</strong> Use the tool for each new client to create unique color identities</li>
                        <li><strong>Brand Guidelines:</strong> Document color schemes for consistent client branding</li>
                        <li><strong>Design Consistency:</strong> Maintain uniform colors across all GHL campaigns and materials</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Color Accessibility & Best Practices</h2>
                    <p>Creating accessible color schemes ensures your designs work for all users:</p>
                    
                    <ul>
                        <li><strong>Contrast Ratios:</strong> Ensure text is readable against background colors</li>
                        <li><strong>Color Blindness:</strong> Avoid relying solely on color to convey information</li>
                        <li><strong>Brand Consistency:</strong> Use consistent colors across all touchpoints</li>
                        <li><strong>Cultural Considerations:</strong> Be aware of color meanings in different cultures</li>
                        <li><strong>Testing:</strong> Always test color combinations with real users</li>
                    </ul>
                </article>

                <article class="content-card">
                    <h2>Advanced Color Strategy</h2>
                    <p>Beyond basic color selection, consider these advanced strategies:</p>
                    
                    <ul>
                        <li><strong>Seasonal Variations:</strong> Adapt color schemes for different seasons or campaigns</li>
                        <li><strong>Industry Standards:</strong> Research color trends in your client's industry</li>
                        <li><strong>Competitor Analysis:</strong> Analyze competitor colors to differentiate your clients</li>
                        <li><strong>A/B Testing:</strong> Test different color schemes to optimize conversions</li>
                        <li><strong>Emotional Mapping:</strong> Map colors to specific emotions and actions you want to evoke</li>
                    </ul>
                </article>
            </div>
        </div>
    </section>

    <!-- Related Tools -->
    <section class="related-tools">
        <div class="container">
            <h2>Related GoHighLevel Tools</h2>
            <div class="tools-grid">
                <a href="funnel-calculator.php" class="tool-link">
                    <div class="tool-icon">📊</div>
                    <h3>Funnel Calculator</h3>
                    <p>Calculate funnel performance and optimize conversions</p>
                </a>
                <a href="content-readability-scorer.php" class="tool-link">
                    <div class="tool-icon">📖</div>
                    <h3>Content Readability Scorer</h3>
                    <p>Analyze and improve your content readability</p>
                </a>
                <a href="ab-testing.php" class="tool-link">
                    <div class="tool-icon">🧪</div>
                    <h3>A/B Testing Calculator</h3>
                    <p>Test different designs and optimize performance</p>
                </a>
            </div>
        </div>
    </section>

    <?php inc('includes/footer.php'); ?>
    <script src="common.js"></script>
    <script>`n    `n    // Load footer`n    fetch("footer.php")`n        .then(response => response.text())`n        .then(data => {`n            document.getElementById("footer-placeholder").innerHTML = data;`n        });`n});`n`n
        // Color Palette Generator functionality
        function generatePalette() {
            const primaryColor = document.getElementById('primaryColor').value;
            const palette = generateColorPalette(primaryColor);
            displayPalette(palette);
        }

        function generateRandomPalette() {
            const randomColor = '#' + Math.floor(Math.random()*16777215).toString(16);
            document.getElementById('primaryColor').value = randomColor;
            document.getElementById('primaryColorHex').value = randomColor;
            const palette = generateColorPalette(randomColor);
            displayPalette(palette);
        }

        function generateAccessiblePalette() {
            const primaryColor = document.getElementById('primaryColor').value;
            const palette = generateAccessibleColors(primaryColor);
            displayPalette(palette);
        }

        function generateColorPalette(baseColor) {
            // Convert hex to HSL
            const hsl = hexToHsl(baseColor);
            
            // Generate complementary colors
            const complementary = hslToHex((hsl.h + 180) % 360, hsl.s, hsl.l);
            const analogous1 = hslToHex((hsl.h + 30) % 360, hsl.s, hsl.l);
            const analogous2 = hslToHex((hsl.h - 30 + 360) % 360, hsl.s, hsl.l);
            const triadic1 = hslToHex((hsl.h + 120) % 360, hsl.s, hsl.l);
            const triadic2 = hslToHex((hsl.h + 240) % 360, hsl.s, hsl.l);
            
            return {
                primary: baseColor,
                complementary: complementary,
                analogous1: analogous1,
                analogous2: analogous2,
                triadic1: triadic1,
                triadic2: triadic2
            };
        }

        function generateAccessibleColors(baseColor) {
            const hsl = hexToHsl(baseColor);
            
            // Generate accessible variations
            const light = hslToHex(hsl.h, hsl.s, Math.min(95, hsl.l + 30));
            const dark = hslToHex(hsl.h, hsl.s, Math.max(15, hsl.l - 30));
            const muted = hslToHex(hsl.h, Math.max(20, hsl.s - 30), hsl.l);
            const vibrant = hslToHex(hsl.h, Math.min(100, hsl.s + 20), hsl.l);
            
            return {
                primary: baseColor,
                light: light,
                dark: dark,
                muted: muted,
                vibrant: vibrant,
                neutral: hslToHex(hsl.h, 0, hsl.l)
            };
        }

        function hexToHsl(hex) {
            const r = parseInt(hex.slice(1, 3), 16) / 255;
            const g = parseInt(hex.slice(3, 5), 16) / 255;
            const b = parseInt(hex.slice(5, 7), 16) / 255;

            const max = Math.max(r, g, b);
            const min = Math.min(r, g, b);
            let h, s, l = (max + min) / 2;

            if (max === min) {
                h = s = 0;
            } else {
                const d = max - min;
                s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
                switch (max) {
                    case r: h = (g - b) / d + (g < b ? 6 : 0); break;
                    case g: h = (b - r) / d + 2; break;
                    case b: h = (r - g) / d + 4; break;
                }
                h /= 6;
            }

            return {
                h: Math.round(h * 360),
                s: Math.round(s * 100),
                l: Math.round(l * 100)
            };
        }

        function hslToHex(h, s, l) {
            h /= 360;
            s /= 100;
            l /= 100;

            const c = (1 - Math.abs(2 * l - 1)) * s;
            const x = c * (1 - Math.abs((h * 6) % 2 - 1));
            const m = l - c / 2;
            let r = 0, g = 0, b = 0;

            if (0 <= h && h < 1) {
                r = c; g = x; b = 0;
            } else if (1 <= h && h < 2) {
                r = x; g = c; b = 0;
            } else if (2 <= h && h < 3) {
                r = 0; g = c; b = x;
            } else if (3 <= h && h < 4) {
                r = 0; g = x; b = c;
            } else if (4 <= h && h < 5) {
                r = x; g = 0; b = c;
            } else if (5 <= h && h <= 6) {
                r = c; g = 0; b = x;
            }

            const rHex = Math.round((r + m) * 255).toString(16).padStart(2, '0');
            const gHex = Math.round((g + m) * 255).toString(16).padStart(2, '0');
            const bHex = Math.round((b + m) * 255).toString(16).padStart(2, '0');

            return `#${rHex}${gHex}${bHex}`;
        }

        function displayPalette(palette) {
            const swatchesContainer = document.getElementById('colorSwatches');
            const infoContainer = document.getElementById('paletteInfo');
            
            let swatchesHTML = '';
            let infoHTML = '<h4>Color Information:</h4><ul>';
            
            for (const [name, color] of Object.entries(palette)) {
                swatchesHTML += `
                    <div class="color-swatch">
                        <div class="color-preview" style="background-color: ${color}"></div>
                        <div class="color-details">
                            <span class="color-name">${name.charAt(0).toUpperCase() + name.slice(1)}</span>
                            <span class="color-hex">${color}</span>
                        </div>
                        <button class="copy-btn" onclick="copyToClipboard('${color}')">Copy</button>
                    </div>
                `;
                
                infoHTML += `<li><strong>${name.charAt(0).toUpperCase() + name.slice(1)}:</strong> ${color}</li>`;
            }
            
            infoHTML += '</ul>';
            
            swatchesContainer.innerHTML = swatchesHTML;
            infoContainer.innerHTML = infoHTML;
            document.getElementById('paletteResult').style.display = 'block';
        }

        function exportPalette() {
            const palette = document.getElementById('colorSwatches').innerHTML;
            const exportData = {
                timestamp: new Date().toISOString(),
                palette: palette
            };
            
            const dataStr = JSON.stringify(exportData, null, 2);
            const dataBlob = new Blob([dataStr], {type: 'application/json'});
            
            const link = document.createElement('a');
            link.href = URL.createObjectURL(dataBlob);
            link.download = 'color-palette.json';
            link.click();
        }

        function copyToClipboard(text) {
            navigator.clipboard.writeText(text).then(function() {
                const button = event.target;
                const originalText = button.textContent;
                button.textContent = 'Copied!';
                button.style.background = '#28a745';
                
                setTimeout(() => {
                    button.textContent = originalText;
                    button.style.background = '';
                }, 2000);
            }).catch(function(err) {
                console.error('Could not copy text: ', err);
                alert('Failed to copy color code. Please copy manually: ' + text);
            });
        }

        // Sync color picker with hex input
        document.getElementById('primaryColor').addEventListener('input', function() {
            document.getElementById('primaryColorHex').value = this.value;
        });

        document.getElementById('primaryColorHex').addEventListener('input', function() {
            if (this.value.match(/^#[0-9A-Fa-f]{6}$/)) {
                document.getElementById('primaryColor').value = this.value;
            }
        });

        // Generate initial palette
        window.addEventListener('load', function() {
            generatePalette();
        });
    </script>

    <style>
        .color-swatches {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }

        .color-swatch {
            display: flex;
            flex-direction: column;
            border: 1px solid #ddd;
            border-radius: 8px;
            overflow: hidden;
            background: white;
        }

        .color-preview {
            height: 80px;
            width: 100%;
        }

        .color-details {
            padding: 15px;
            text-align: center;
        }

        .color-name {
            display: block;
            font-weight: 600;
            margin-bottom: 5px;
            color: #333;
        }

        .color-hex {
            display: block;
            font-family: monospace;
            color: #666;
            font-size: 14px;
        }

        .copy-btn {
            background: #0a2038;
            color: white;
            border: none;
            padding: 10px;
            cursor: pointer;
            transition: background 0.3s;
        }

        .copy-btn:hover {
            background: #1a4060;
        }

        .palette-info {
            margin: 20px 0;
            padding: 20px;
            background: #f8f9fa;
            border-radius: 8px;
        }

        .palette-info ul {
            list-style: none;
            padding: 0;
        }

        .palette-info li {
            padding: 5px 0;
            border-bottom: 1px solid #eee;
        }

        .palette-info li:last-child {
            border-bottom: none;
        }
    </style>
</body>
</html>




